program PREDICTORCORRECTOR;
{--------------------------------------------------------------------}
{  Alg9'6-8.pas   Pascal program for implementing Algorithm 9.6-8    }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 9.6 (Adams-Bashforth-Moulton Method).                   }
{  Section   9.6, Predictor-Corrector Method, Page 471               }
{                                                                    }
{  Algorithm 9.7 (Milne-Simpson Method).                             }
{  Section   9.6, Predictor-Corrector Method, Page 472               }
{                                                                    }
{  Algorithm 9.8 (The Hamming Method).                               }
{  Section   9.6, Predictor-Corrector Method, Page 473               }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 630;
    MaxM = 630;
    FunMax = 9;

  type
    VECTOR = array[0..MaxM] of real;
    LETTERS = string[200];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);

  var
    FunType, GNpts, Inum, M, Mend, Meth, Sub: integer;
    A, B, Y0, Rnum: real;
    Ans: CHAR;
    T, X, Y: VECTOR;
    Mess: LETTERS;
    State: States;
    DoMo: DoSome;

  function F (T, Y: real): real;
  begin
    case FunType of
      1: 
        F := (T - Y) / 2;
      2: 
        F := (Y - T) / 2;
      3: 
        F := T * T - Y;
      4: 
        F := 3 * T + 3 * Y;
      5: 
        F := -T * Y;
      6:
        F := EXP(-2 * T) - 2 * Y;
      7: 
        F := 2 * T * Y * Y;
      8: 
        F := 1 + Y * Y;
      9: 
        F := T * T + Y * Y;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1: 
        WRITELN('Y` = (T-Y)/2');
      2: 
        WRITELN('Y` = (Y-T)/2');
      3: 
        WRITELN('Y` = T^2 - Y');
      4: 
        WRITELN('Y` = 3*T + 3*Y');
      5:
        WRITELN('Y` = -T*Y');
      6: 
        WRITELN('Y` = EXP(-2*T) - 2*Y');
      7: 
        WRITELN('Y` = 2*T*Y^2');
      8: 
        WRITELN('Y` = 1 + Y^2');
      9: 
        WRITELN('Y` = T^2 + Y^2');
    end;
  end;

  procedure ABM ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E7;
    var
      K: integer;
      H, H2, F0, F1, F2, F3, F4, P, TK, YK: real;
    procedure POLE;                              {Euler or Heun's method}
      label
        990;
      var
        K1, K2, P: real;
    begin
      K1 := H * F(TK, YK);
      P := YK + K1;
      T[K + 1] := A + H * (K + 1);
      if Big < ABS(P) then
        begin
          Y[K + 1] := P;
          goto 990;
        end;
      K2 := F(T[K + 1], P);
      Y[K + 1] := YK + H * (K1 + K2) / 2;
990:
    end;                                        {End of Procedure Pole}
    procedure RUNGEKUTTA;
      var
        J: integer;
        K1, K2, K3, K4, TJ, YJ: real;
    begin                                      {Start with Runge-Kutta}
      for J := 0 to 2 do
        begin
          TJ := T[J];
          YJ := Y[J];
          K1 := H * F(TJ, YJ);
          K2 := H * F(TJ + H / 2, YJ + 0.5 * K1);
          K3 := H * F(TJ + H / 2, YJ + 0.5 * K2);
          K4 := H * F(TJ + H, YJ + K3);
          Y[J + 1] := YJ + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
          T[J + 1] := A + H * (J + 1);
        end;
    end;                                 {End of Procedure Runge-Kutta}
  begin                                        {Start of Procedure ABM}
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    RUNGEKUTTA;
    F0 := F(T[0], Y[0]);
    F1 := F(T[1], Y[1]);
    F2 := F(T[2], Y[2]);
    F3 := F(T[3], Y[3]);
    H2 := H / 24;
    for K := 3 to M - 1 do
      begin
        TK := T[K];
        YK := Y[K];
        P := YK + H2 * (-9 * F0 + 37 * F1 - 59 * F2 + 55 * F3);
        if Big < ABS(P) then
          begin
            POLE;
            goto 999;
          end;
        T[K + 1] := A + H * (K + 1);
        F4 := F(T[K + 1], P);
        Y[K + 1] := YK + H2 * (F1 - 5 * F2 + 19 * F3 + 9 * F4);
        if Big < ABS(Y[K + 1]) then
          begin
            POLE;
            goto 999;
          end;
        F0 := F1;
        F1 := F2;
        F2 := F3;
        F3 := F(T[K + 1], Y[K + 1]);
      end;
999:
    Mend := K + 1;
  end;                                           {End of Procedure ABM}

  procedure MILNE ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E7;
    var
      K: integer;
      H, F1, F2, F3, F4, Pmod, Pnew, Pold, TK, YK, Yold: real;
    procedure POLE;                              {Euler or Heun's method}
      label
        990;
      var
        K1, K2, P: real;
    begin
      TK := T[K];
      YK := Y[K];
      K1 := H * F(TK, YK);
      P := YK + K1;
      T[K + 1] := A + H * (K + 1);
      if Big < ABS(P) then
        begin
          Y[K + 1] := P;
          goto 990;
        end;
      K2 := F(T[K + 1], P);
      Y[K + 1] := YK + H * (K1 + K2) / 2;
990:
    end;                                        {End of Procedure Pole}
    procedure RUNGEKUTTA;
      var
        J: integer;
        K1, K2, K3, K4, TJ, YJ: real;
    begin                                      {Start with Runge-Kutta}
      for J := 0 to 2 do
        begin
          TJ := T[J];
          YJ := Y[J];
          K1 := H * F(TJ, YJ);
          K2 := H * F(TJ + H / 2, YJ + 0.5 * K1);
          K3 := H * F(TJ + H / 2, YJ + 0.5 * K2);
          K4 := H * F(TJ + H, YJ + K3);
          Y[J + 1] := YJ + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
          T[J + 1] := A + H * (J + 1);
        end;
    end;                                 {End of Procedure Runge-Kutta}
  begin                                      {Start of Procedure Milne}
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    RUNGEKUTTA;
    F1 := F(T[1], Y[1]);
    F2 := F(T[2], Y[2]);
    F3 := F(T[3], Y[3]);
    Pold := 0;
    Yold := 0;
    for K := 3 to M - 1 do
      begin
        Pnew := Y[K - 3] + 4 * H * (2 * F1 - F2 + 2 * F3) / 3;
        Pmod := Pnew + 28 * (Yold - Pold) / 29;
        T[K + 1] := A + H * (K + 1);
        if Big < ABS(Pmod) then
          begin
            POLE;
            goto 999;
          end;
        F4 := F(T[K + 1], Pmod);
        Y[K + 1] := Y[K - 1] + H * (F2 + 4 * F3 + F4) / 3;
        if Big < ABS(Y[K + 1]) then
          begin
            POLE;
            goto 999;
          end;
        Pold := Pnew;
        Yold := Y[K + 1];
        F1 := F2;
        F2 := F3;
        F3 := F(T[K + 1], Y[K + 1]);
      end;
999:
    Mend := K + 1;
  end;                                         {End of Procedure Milne}

  procedure HAMMING ({FUNCTION F(T,Y:real): real;}
                  A, B, Y0: real; M: integer; var Mend: integer; var T, Y: VECTOR);
    label
      999;
    const
      Big = 1E7;
    var
      K: integer;
      Cnew, Cold, H, F1, F2, F3, F4, Pmod, Pnew, Pold, TK, YK: real;
    procedure POLE;                              {Euler or Heun's method}
      label
        990;
      var
        K1, K2, P: real;
    begin
      TK := T[K];
      YK := Y[K];
      K1 := H * F(TK, YK);
      P := YK + K1;
      T[K + 1] := A + H * (K + 1);
      if Big < ABS(P) then
        begin
          Y[K + 1] := P;
          goto 990;
        end;
      K2 := F(T[K + 1], P);
      Y[K + 1] := YK + H * (K1 + K2) / 2;
990:
    end;                                        {End of Procedure Pole}
    procedure RUNGEKUTTA;
      var
        J: integer;
        K1, K2, K3, K4, TJ, YJ: real;
    begin                                        {Start with Runge-Kutta}
      for J := 0 to 2 do
        begin
          TJ := T[J];
          YJ := Y[J];
          K1 := H * F(TJ, YJ);
          K2 := H * F(TJ + H / 2, YJ + 0.5 * K1);
          K3 := H * F(TJ + H / 2, YJ + 0.5 * K2);
          K4 := H * F(TJ + H, YJ + K3);
          Y[J + 1] := YJ + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
          T[J + 1] := A + H * (J + 1);
        end;
    end;                                 {End of Procedure Runge-Kutta}
  begin                                    {Start of Procedure Hamming}
    H := (B - A) / M;
    T[0] := A;
    Y[0] := Y0;
    RUNGEKUTTA;
    F1 := F(T[1], Y[1]);
    F2 := F(T[2], Y[2]);
    F3 := F(T[3], Y[3]);
    Pold := 0;
    Cold := 0;
    for K := 3 to M - 1 do
      begin
        Pnew := Y[K - 3] + 4 * H * (2 * F1 - F2 + 2 * F3) / 3;
        Pmod := Pnew + 112 * (Cold - Pold) / 121;
        T[K + 1] := A + H * (K + 1);
        if Big < ABS(Pmod) then
          begin
            POLE;
            goto 999;
          end;
        F4 := F(T[K + 1], Pmod);
        Cnew := (9 * Y[K] - Y[K - 2] + 3 * H * (-F2 + 2 * F3 + F4)) / 8;
        Y[K + 1] := Cnew + 9 * (Pnew - Cnew) / 121;
        if Big < ABS(Y[K + 1]) then
          begin
            POLE;
            goto 999;
          end;
        Pold := Pnew;
        Cold := Cnew;
        F1 := F2;
        F2 := F3;
        F3 := F(T[K + 1], Y[K + 1]);
      end;
999:
    Mend := K + 1;
  end;                                       {End of Procedure Hamming}

  procedure MESSAGE (var Meth: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('                      PREDICTOR-CORRECTOR  METHODS');
    WRITELN;
    WRITELN;
    WRITELN('          Solution of the differential equation  y` = F(t,y)');
    WRITELN;
    WRITELN('          with the initial condition  y(a) = y .');
    WRITELN('                                              0');
    WRITELN;
    WRITELN('          A numerical approximation is computed over  [a,b].');
    WRITELN;
    WRITELN;
    WRITELN('                      Choose the method of approximation:');
    WRITELN;
    WRITELN;
    WRITELN('                < 1 > Adams-Bashforth-Moulton method');
    WRITELN;
    WRITELN('                < 2 > Milne-Simpson`s method');
    WRITELN;
    WRITELN('                < 3 > Hamming`s method');
    WRITELN;
    WRITELN;
    Mess := '                      SELECT < 1 - 3 > ?  ';
    Meth := 1;
    WRITELN;
    WRITE(Mess);
    READLN(Meth);
    if (Meth < 1) and (State <> Changes) then
      Meth := 1;
    if (Meth > 3) and (State <> Changes) then
      Meth := 3;
  end;

  procedure GETPOINTS (var FunType: integer; var A, B, Y0: real; var M: integer; MaxM: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITE('       You chose ');
    case Meth of
      1: 
        WRITE('the Adams-Bashforth-Moulton method');
      2: 
        WRITE('the Milne-Simpson method');
      3: 
        WRITE('Hamming`s method');
    end;
    WRITELN(' to solve Y` = F(T,Y).');
    WRITELN;
    WRITELN('       Choose your D.E.:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('  <', K : 1, '>  ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    WRITELN;
    WRITE('       SELECT < 1 - ', FunMax : 1, ' > ?   ');
    FunType := 1;
    WRITELN;
    READLN(FunType);
    if (FunType < 1) and (State <> Changes) then
      FunType := 1;
    if (FunType > FunMax) and (State <> Changes) then
      FunType := FunMax;
    CLRSCR;
    WRITE('     You chose ');
    case Meth of
      1: 
        WRITE('the Adams-Bashforth-Moulton method');
      2: 
        WRITE('the Milne-Simpson method');
      3: 
        WRITE('Hamming`s method');
    end;
    WRITELN(' to solve the D.E.');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('     with the initial condition Y(A) = Y .');
    WRITELN('                                        0');
    WRITELN;
    WRITELN('         A numerical approximation is computed over [A,B].');
    WRITELN;
    WRITELN('     You must enter the endpoints for the interval, the');
    WRITELN;
    WRITELN('     initial condition Y(A) = Y , and the number of steps M.');
    WRITELN('                               0');
    WRITELN;
    WRITELN;
    WRITE('         Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure EPOINTS (var A, B, Y0: real; var M: integer; var State: STATES; MaxM, Meth: integer);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      I: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        CLRSCR;
        WRITELN;
        WRITE('                 ');
        PRINTFUNCTION(FunType);
        WRITELN;
        WRITELN;
        if (Stat = Enter) then
          begin
            Mess := '     ENTER  the  left  endpoint  A = ';
            WRITELN;
            WRITE(Mess);
            READLN(A);
            WRITELN;
            Mess := '     ENTER  the  right endpoint  B = ';
            WRITELN;
            WRITE(Mess);
            READLN(B);
            WRITELN;
            Mess := '     ENTER initial condition  Y(A) = ';
            WRITELN;
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            Mess := '     ENTER the number of steps   M = ';
            M := 4;
            WRITELN;
            WRITE(Mess);
            READLN(M);
            WRITELN;
            if M < 4 then
              M := 4;
            if M > MaxM then
              M := Maxm;
          end
        else
          begin
            WRITELN('     The  left  endpoint  is     A =', A : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The  right endpoint  is     B =', B : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   condition  is  Y(A) =', Y0 : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The number of steps  is     M =  ', M);
          end;
        WRITELN;
        WRITELN;
        WRITE('     Want to make a change ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            CLRSCR;
            WRITELN;
            WRITE('                 ');
            PRINTFUNCTION(FunType);
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITELN;
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITELN;
            WRITE(Mess);
            READLN(B);
            WRITELN;
            WRITELN('     The   current   I. C.  is  Y(A) =', Y0 : 15 : 7);
            Mess := '     Now  ENTER the NEW  I. C.  Y(A) = ';
            WRITELN;
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            WRITELN('     The  current value of  M  is  M =  ', M);
            Mess := '     Now  ENTER  the NEW value of  M = ';
            WRITELN;
            WRITE(Mess);
            READLN(M);
            if (M < 4) then
              M := 4;
            if (M > MaxM) then
              M := MaxM;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (T, Y: VECTOR; M, Mend: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    case Meth of
      1: 
        WRITELN('The Adams-Bashforth-Moulton method was used to solve');
      2: 
        WRITELN('Milne-Simpson method was used to solve');
      3: 
        WRITELN('Hamming`s method was used to solve');
    end;
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('with  Y(', T[0] : 15 : 7, '  ) =', Y[0] : 15 : 7);
    WRITELN;
    WRITELN('     ', '           T             ', '      Y');
    WRITELN('    k', '            k            ', '       k');
    WRITELN('  ------------------------------------------------');
    WRITELN;
    for K := 0 to Mend do
      begin
        WRITELN(K : 5, '   ', T[K] : 15 : 7, '     ', Y[K] : 15 : 7);
        WRITELN;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN;
          end;
      end;
    if Mend < M then
      begin
        WRITELN('The solution points are approaching a pole.');
        WRITELN;
      end;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  A := 0;
  B := 1;
  Y0 := 0;
  M := 4;
  State := Working;
  while Meth <> 0 do
    begin
      MESSAGE(Meth);
      DoMo := Go;
      while DoMo = Go do
        begin
          GETPOINTS(FunType, A, B, Y0, M, MaxM);
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, Y0, M, State, MaxM, Meth);
              case Meth of
                1: 
                  ABM(A, B, Y0, M, Mend, T, Y);
                2: 
                  MILNE(A, B, Y0, M, Mend, T, Y);
                3: 
                  HAMMING(A, B, Y0, M, Mend, T, Y);
              end;
              RESULTS(T, Y, M, Mend);
              WRITELN;
              WRITELN;
              WRITE('Want to try  a different  initial condition ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done
              else
                State := Changes;
            end;
          WRITELN;
          WRITE('Want  to change  the  differential equation ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop
          else
            State := Changes;
        end;
      WRITELN;
      WRITE('Want to try another method of approximation ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Meth := 0
      else
        State := Changes;
    end;
end.                                            {End of Main Program}

